<?php
/**
 * サニタイズ関連
 *
 * @author     AnimaGate, Inc.
 * @link       https://support.animagate.com/product/wp-nishiki-pro/
 */

/**
 * テキスト(wp_filter_nohtml_kses)
 *
 * @param string $text 文字列
 * @return string
 */
function nishiki_pro_sanitize_text( $text ) {
	return sanitize_text_field( $text );
}

/**
 * テキストエリア Text Area(Allow HTML)
 *
 * @param string $text 文字列
 * @return string
 */
function nishiki_pro_sanitize_textarea( $text ) {
	$allowed_html = array(
		'a'      => array(
			'href'    => array(),
			'onclick' => array(),
			'target'  => array(),
		),
		'br'     => array(),
		'strong' => array(),
		'b'      => array(),
	);

	return wp_kses( $text, $allowed_html );
}

/**
 * テキストエリア Text Area(Allow HTML)
 *
 * @param string $text 文字列
 * @return string
 */
function nishiki_pro_sanitize_textarea_script( $text ) {

	$allowed_html = array(
		'script' => array(
			'async' => true,
			'src'   => true,
			'type'  => true
		),
		'noscript' => array(),
		'iframe'   => array(
			'src'               => true,
			'height'            => true,
			'width'             => true,
			'style'             => true,
			'frameborder'       => true,
			'allowfullscreen'   => true
		),
		'div' => array(
			'id'    => true,
			'style' => true
		)
	);

	// フィルター
	$filtered_code = wp_kses( $text, $allowed_html );

	$filter_patterns = [
		'/alert\s*\(/i',         // alert()
		'/eval\s*\(/i',          // eval()
		'/document\.write\s*\(/i', // document.write()
		'/on\w+\s*=/i',          // onclick, onmouseover など
	];
	
	$filtered_code = preg_replace( $filter_patterns, '', $filtered_code );	 

	// <script> タグ内に src 属性がないものを削除（インラインJS禁止）
	$filtered_code = preg_replace('/<script(?![^>]*\ssrc=["\']).*?<\/script>/is', '', $filtered_code);

	// "javascript:" を含む文字列を除去
	$filtered_code = preg_replace('/javascript\s*:/i', '', $filtered_code);

	return $filtered_code;
}

/**
 * シェアボタン出力
 *
 * @param string $text 文字列
 * @return void
 */
function nishiki_pro_sanitize_share_button( $text ) {
	$allowed_tags = wp_kses_allowed_html( 'post' );

	// 追加許可タグ
	$add_allowed_tags = array(
		'a' => array(
			'onclick' => true,
		),
	);

	$allowed_tags['a'] += $add_allowed_tags['a'];

	echo wp_kses( $text, $allowed_tags );
}

/**
 * Checkbox
 *
 * @param int $checked チェックボックスの値
 * @return boolean
 */
function nishiki_pro_sanitize_checkbox( $checked ) {
	return ( ( isset( $checked ) && true === $checked ) ? true : false );
}

/**
 * Select(Sidebar)
 *
 * @param string $input セレクトボックスの値
 * @return string
 */
function nishiki_pro_sanitize_choices( $input ) {
	$valid = array( 'left', 'right', 'bottom', 'none' );
	if ( in_array( $input, $valid, true ) ) {
		return $input;
	}

	return 'none';
}

/**
 * Header Layout
 *
 * @param string $input セレクトボックスの値
 * @return string
 */
function nishiki_pro_sanitize_choices_header_layout( $input ) {
	$valid = array( 'default', 'center' );
	if ( in_array( $input, $valid, true ) ) {
		return $input;
	}
	return 'default';
}

/**
 * Background Blur
 *
 * @param string $input セレクトボックスの値
 * @return string
 */
function nishiki_pro_sanitize_choices_background_blur( $input ) {
	$valid = array( 'panel-blur-sm', 'panel-blur-md', 'panel-blur-lg' );
	if ( in_array( $input, $valid, true ) ) {
		return $input;
	}
	return '';
}

/**
 * Background Blur
 *
 * @param string $input セレクトボックスの値
 * @return string
 */
function nishiki_pro_sanitize_choices_pseudo_background_blur( $input ) {
	$valid = array( 'pseudo-backdrop-blur-sm', 'pseudo-backdrop-blur-md', 'pseudo-backdrop-blur-lg' );
	if ( in_array( $input, $valid, true ) ) {
		return $input;
	}
	return '';
}

/**
 * Background Blur
 *
 * @param string $input セレクトボックスの値
 * @return string
 */
function nishiki_pro_sanitize_choices_panel_open_type( $input ) {
	$valid = array( 'panel-slide', 'panel-fade' );
	if ( in_array( $input, $valid, true ) ) {
		return $input;
	}
	return '';
}

/**
 * Campaign Layout
 *
 * @param string $input セレクトボックスの値
 * @return string
 */
function nishiki_pro_sanitize_choices_campaign_layout( $input ) {
	$valid = array( 'text-button', 'button-text' );
	if ( in_array( $input, $valid, true ) ) {
		return $input;
	}
	return 'text-button';
}

/**
 * Columns
 *
 * @param string $input セレクトボックスの値
 * @return string
 */
function nishiki_pro_sanitize_choices_columns( $input ) {
	$valid = array( '1', '2', '3' );
	if ( in_array( $input, $valid, true ) ) {
		return $input;
	}
	return '3';
}

/**
 * Article Design
 *
 * @param string $input セレクトボックスの値
 * @return string
 */
function nishiki_pro_sanitize_choices_archive_article_design( $input ) {
	$valid = array( '', 'border', 'shadow-sm', 'shadow-md', 'shadow-lg', 'shadow-xl', 'shadow-2xl' );
	if ( in_array( $input, $valid, true ) ) {
		return $input;
	}
	return '';
}

/**
 * Order
 *
 * @param string $input セレクトボックスの値
 * @return string
 */
function nishiki_pro_sanitize_choices_order( $input ) {
	$valid = array( 'ASC', 'DESC' );
	if ( in_array( $input, $valid, true ) ) {
		return $input;
	}
	return 'DESC';
}

/**
 * Orderby
 *
 * @param string $input セレクトボックスの値
 * @return string
 */
function nishiki_pro_sanitize_choices_orderby( $input ) {
	$valid = array( 'date', 'modified', 'ID', 'author', 'title', 'name', 'type', 'rand' );
	if ( in_array( $input, $valid, true ) ) {
		return $input;
	}
	return 'date';
}

/**
 * Share Button
 *
 * @param string $input セレクトボックスの値
 * @return string
 */
function nishiki_pro_sanitize_choices_share( $input ) {
	$valid = array( 'top', 'bottom', 'both', 'none' );
	if ( in_array( $input, $valid, true ) ) {
		return $input;
	}
	return 'none';
}

/**
 * Share Button Design Pattern
 *
 * @param string $input セレクトボックスの値
 * @return string
 */
function nishiki_pro_sanitize_choices_share_design_pattern( $input ) {
	$valid = array( '01', '02', '03' );
	if ( in_array( $input, $valid, true ) ) {
		return $input;
	}
	return 'none';
}

/**
 * Social Account Design Pattern
 *
 * @param string $input セレクトボックスの値
 * @return string
 */
function nishiki_pro_sanitize_choices_social_design_pattern( $input ) {
	$valid = array( '01', '02', '03' );
	if ( in_array( $input, $valid, true ) ) {
		return $input;
	}
	return 'none';
}

/**
 * Global Nav Design Pattern
 *
 * @param string $input セレクトボックスの値
 * @return string
 */
function nishiki_pro_sanitize_choices_global_nav_design_pattern( $input ) {
	$valid = array( '01', '02', '03' );
	if ( in_array( $input, $valid, true ) ) {
		return $input;
	}
	return 'none';
}

/**
 * Header Button Design Pattern
 *
 * @param string $input セレクトボックスの値
 * @return string
 */
function nishiki_pro_sanitize_choices_header_button_pattern( $input ) {
	$valid = array( '01', '02' );
	if ( in_array( $input, $valid, true ) ) {
		return $input;
	}
	return '01';
}

/**
 * Header Global Nav Open Type
 *
 * @param string $input セレクトボックスの値
 * @return string
 */
function nishiki_pro_sanitize_choices_header_global_nav_open_type( $input ) {
	$valid = array( 'click', 'hover' );
	if ( in_array( $input, $valid, true ) ) {
		return $input;
	}
	return 'click';
}

/**
 * Footer nav
 *
 * @param string $input セレクトボックスの値
 * @return string
 */
function nishiki_pro_sanitize_choices_footer_nav_sticky( $input ) {
	$valid = array( 'sticky', 'mobile-sticky', 'none' );
	if ( in_array( $input, $valid, true ) ) {
		return $input;
	}
	return 'none';
}

/**
 * TOC
 *
 * @param string $input セレクトボックスの値
 * @return string
 */
function nishiki_pro_sanitize_choices_toc_min_level( $input ) {
	$valid = array( '2', '3', '4', '5', '6' );
	if ( in_array( $input, $valid, true ) ) {
		return $input;
	}
	return 3;
}

/**
 * Item
 *
 * @param string $input セレクトボックスの値
 * @return string
 */
function nishiki_pro_sanitize_choices_item( $input ) {
	$valid = array( 'disabled', 'icon', 'image' );
	if ( in_array( $input, $valid, true ) ) {
		return $input;
	}
	return 'disabled';
}

/**
 * Section
 *
 * @param string $input セレクトボックスの値
 * @return string
 */
function nishiki_pro_sanitize_choices_front_page_section( $input ) {
	$valid = array( 'disabled', 'recently', 'custom' );
	if ( in_array( $input, $valid, true ) ) {
		return $input;
	}
	return 'disabled';
}

/**
 * Featured Items
 *
 * @param string $input セレクトボックスの値
 * @return string
 */
function nishiki_pro_sanitize_choices_front_page_featured_items( $input ) {
	$valid = array( 'disabled', 'enabled' );
	if ( in_array( $input, $valid, true ) ) {
		return $input;
	}
	return 'disabled';
}

/**
 * Text Align
 *
 * @param string $input セレクトボックスの値
 * @return string
 */
function nishiki_pro_sanitize_choices_align( $input ) {
	$valid = array( 'left', 'center', 'right' );
	if ( in_array( $input, $valid, true ) ) {
		return $input;
	}
	return 'center';
}

/**
 * PR Label Type
 *
 * @param string $input セレクトボックスの値
 * @return string
 */
function nishiki_pro_sanitize_choices_pr_type( $input ) {
	$valid = array( 'fill', 'outline' );
	if ( in_array( $input, $valid, true ) ) {
		return $input;
	}
	return 'fill';
}

/**
 * Number
 *
 * @param int    $number 数値
 * @param object $setting 設定
 * @return int
 */
function nishiki_pro_sanitize_number( $number, $setting ) {
	$number = absint( $number );

	return ( $number ? $number : $setting->default );
}

/**
 * Number range
 *
 * @param int    $number 数値
 * @param object $setting 設定
 * @return int
 */
function nishiki_pro_sanitize_number_range( $number, $setting ) {
	$number = absint( $number );
	$atts   = ( isset( $setting->manager->get_control( $setting->id )->input_attrs ) ? $setting->manager->get_control( $setting->id )->input_attrs : array() );
	$min    = ( isset( $atts['min'] ) ? $atts['min'] : $number );
	$max    = ( isset( $atts['max'] ) ? $atts['max'] : $number );
	$step   = ( isset( $atts['step'] ) ? $atts['step'] : 1 );
	return ( $min <= $number && $number <= $max && is_int( $number / $step ) ? $number : $setting->default );
}

/**
 * File uploader
 *
 * @param string $image ファイル名
 * @param object $setting 設定
 * @return string
 */
function nishiki_pro_sanitize_image( $image, $setting ) {
	$mimes = array(
		'jpg|jpeg|jpe' => 'image/jpeg',
		'gif'          => 'image/gif',
		'png'          => 'image/png',
		'bmp'          => 'image/bmp',
		'tif|tiff'     => 'image/tiff',
		'ico'          => 'image/x-icon',
		'avif'         => 'image/avif',
		'webp'         => 'image/webp',
	);
	$file  = wp_check_filetype( $image, $mimes );
	return ( $file['ext'] ? $image : $setting->default );
}

/**
 * Meta Catd Type
 *
 * @param string $input セレクトボックスの値
 * @return string
 */
function nishiki_pro_sanitize_choices_meta_cards_type( $input ) {
	$valid = array( 'summary', 'summary_large_image' );
	if ( in_array( $input, $valid, true ) ) {
		return $input;
	}
	return 'summary';
}

/**
 * JSON-LD Type
 *
 * @param string $input セレクトボックスの値
 * @return string
 */
function nishiki_pro_sanitize_choices_meta_json_ld_type( $input ) {
	$valid = array( 'BlogPosting', 'Article' );
	if ( in_array( $input, $valid, true ) ) {
		return $input;
	}
	return 'BlogPosting';
}

/**
 * Website Type
 *
 * @param string $input セレクトボックスの値
 * @return string
 */
function nishiki_pro_sanitize_choices_meta_json_ld_website_type( $input ) {
	$valid = array( '', 'Organization', 'Person' );
	if ( in_array( $input, $valid, true ) ) {
		return $input;
	}
	return '';
}

/**
 * JSON-LD(contactoption)
 *
 * @param string $input セレクトボックスの値
 * @return string
 */
function nishiki_pro_sanitize_choices_meta_json_ld_contactpoint_contactoption( $input ) {
	$valid = array( '', 'TollFree', 'HearingImpairedSupported' );
	if ( in_array( $input, $valid, true ) ) {
		return $input;
	}
	return '';
}

/**
 * JSON-LD(contactType)
 *
 * @param string $input セレクトボックスの値
 * @return string
 */
function nishiki_pro_sanitize_choices_meta_json_ld_contactpoint_contacttype( $input ) {
	$valid = array(
		'',
		'customer service',
		'technical support',
		'billing support',
		'bill payment',
		'sales',
		'reservations',
		'credit card_support',
		'emergency',
		'baggage tracking',
		'roadside assistance',
		'package tracking',
	);
	if ( in_array( $input, $valid, true ) ) {
		return $input;
	}
	return '';
}

/**
 * Heading choice
 *
 * @param string $input セレクトボックスの値
 * @return string
 */
function nishiki_pro_sanitize_choices_heading_type( $input ) {
	$valid = array(
		'simple',
		'underline',
		'baloon',
		'box',
		'stitch-box',
		'dot-box',
		'polka-dot-box',
		'stripe-box',
		'houndstooth-box',
		'jigsaw-puzzle-box',
		'graph-paper-box',
		'brick-wall-box',
		'sailor-anchor-box',
		'none',
	);
	if ( in_array( $input, $valid, true ) ) {
		return $input;
	}
	return '';
}

/**
 * Campaign choice
 *
 * @param string $input セレクトボックスの値
 * @return string
 */
function nishiki_pro_sanitize_choices_campaign_background_type( $input ) {
	$valid = array(
		'single-color',
		'dot',
		'polka-dot',
		'stripe',
		'check',
	);
	if ( in_array( $input, $valid, true ) ) {
		return $input;
	}
	return '';
}

/**
 * Radio button
 *
 * @param string $input ラジオボタンの値
 * @return string
 */
function nishiki_pro_sanitize_radio( $input ) {
	return sanitize_key( $input );
}
