<?php

if ( ! class_exists( 'NISHIKI_PRO_SPDB_IMPORT_DEMO_REST_API' ) ) {
	/**
	 * Rest API 設定
	 */
	class NISHIKI_PRO_SPDB_IMPORT_DEMO_REST_API {
		/**
		 * 初期設定
		 */
		public function __construct() {
			add_action( 'rest_api_init', array( $this, 'create_endpoint' ) );
		}

		/**
		 * エンドポイント作成
		 *
		 * @return void
		 */
		public function create_endpoint() {
			register_rest_route(
				'import-demo/v1',
				'/init',
				array(
					array(
						'methods'             => 'GET',
						'callback'            => array( $this, 'get_callback' ),
						'permission_callback' => '__return_true',
					),
					array(
						'methods'             => 'PUT',
						'callback'            => array( $this, 'update_callback' ),
						'permission_callback' => array( $this, 'rest_permission' ),
					),
				)
			);
		}

		/**
		 * コールバック（アップデート）
		 *
		 * @param array $req リクエスト
		 * @return object
		 */
		public function get_callback() {
			$data = array(
				'message' => 'REST API Ready',
			);

			return new WP_REST_Response( $data, 200 );
		}

		/**
		 * コールバック（アップデート）
		 *
		 * @param array $req リクエスト
		 * @return object
		 */
		public function update_callback( $req ) {
			if ( ! isset( $req ) ) {
				return false;
			}

			if( ! isset( $req['demo_site_name'] ) ){
				return false;
			}

			// 一旦全削除
			$this->delete_all_data();

			// デモデータインポート
			$this->import_demo_data( $req );

			return new WP_REST_Response( $req, 200 );
		}

		/**
		 * アクセスできる権限
		 *
		 * @return boolean
		 */
		public function rest_permission() {
			return current_user_can( 'manage_options' );
		}

		public function import_demo_data( $req ){
	        $request = wp_remote_get( 'https://demo.animagate.com/' . $req['demo_site_name'] . '/wp-json/demo-site/v1/settings', [ 'sslverify' => false ] );
			$restponse = wp_remote_retrieve_body( $request );
			$data = json_decode( $restponse, true );

			// theme_mods
			$this->import_theme_mods( $data['theme_mods'] );

			// logo
			$this->import_logo( $data['logo_url'] );

			// taxonomies
			$this->import_taxonomies( $data[ 'taxonomies' ] );

			// posts
			$this->import_posts( $data );

			// pages
			$this->import_pages( $data );

			// nishiki_pro_content
			$this->import_nishiki_pro_content( $data['theme_options'] );

			// nav_menus
			$this->import_nav_menus( $data['export_nav'] );

			// widgets
			$this->import_widgets( $data['widgets'] );

			// theme_options
			$this->import_theme_options( $data );			
		}

		public function import_theme_mods( $data ){
			foreach ( $data as $key => $value ) {
				set_theme_mod( $key, $value );
			}
		}

		public function import_logo( $logo_url ) {
			if ( empty( $logo_url ) ) {
				return;
			}

			require_once ABSPATH . 'wp-admin/includes/file.php';
			require_once ABSPATH . 'wp-admin/includes/media.php';
			require_once ABSPATH . 'wp-admin/includes/image.php';

			// 画像をメディアに追加
			$tmp = download_url( $logo_url );

			if ( is_wp_error( $tmp ) ) {
				return;
			}

			$file_array = array(
				'name'     => basename( $logo_url ),
				'tmp_name' => $tmp
			);

			$id = media_handle_sideload( $file_array, 0 );

			// エラーがなければテーマカスタマイザーのロゴに設定
			if ( ! is_wp_error( $id ) ) {
				set_theme_mod( 'custom_logo', $id );
			}
		}

		public function import_taxonomies( $data ){
			$existing_taxonomies = get_taxonomies([], 'names');

			foreach ( $data as $taxonomy => $terms ) {
				if (!in_array($taxonomy, $existing_taxonomies, true)) {
					// 存在しなければスキップ
					continue;
				}

				foreach ($terms as $term) {
					$exists = get_term_by('slug', $term['slug'], $taxonomy);
					if (!$exists) {
						wp_insert_term($term['name'], $taxonomy, [
							'slug'        => $term['slug'],
							'description' => $term['description'],
							'parent'      => $term['parent'],
						]);
					}
				}
			}
		}

		public function import_posts($data) {
			// --- 旧カテゴリID → slug マップ ---
			$old_cat_id_to_slug = [];
			foreach ($data['taxonomies']['category'] ?? [] as $cat) {
				if (isset($cat['term_id'], $cat['slug'])) {
					$old_cat_id_to_slug[(string)$cat['term_id']] = $cat['slug'];
				}
			}

			// --- 移行先カテゴリ slug → 新ID マップ ---
			$cat_slug_to_id = [];
			foreach ($data['taxonomies']['category'] ?? [] as $cat) {
				if (!isset($cat['slug'])) continue;
				$term = term_exists($cat['slug'], 'category');
				if (!$term) {
					$term = wp_insert_term($cat['name'], 'category', [
						'slug' => $cat['slug'],
						'description' => $cat['description'] ?? '',
						'parent' => $cat['parent'] ?? 0,
					]);
				}
				if (!is_wp_error($term)) {
					$cat_slug_to_id[$cat['slug']] = (int)$term['term_id'];
				}
			}

			// --- 旧タグID → slug マップ ---
			$old_tag_id_to_slug = [];
			// --- 移行先タグ slug → 新ID マップ ---
			$tag_slug_to_id = [];
			foreach ($data['taxonomies']['post_tag'] ?? [] as $tag) {
				if (isset($tag['term_id'], $tag['slug'])) {
					$old_tag_id_to_slug[(string)$tag['term_id']] = $tag['slug'];
				}
				$term = term_exists($tag['slug'], 'post_tag');
				if (!$term) {
					$term = wp_insert_term($tag['name'], 'post_tag', [
						'slug' => $tag['slug'],
						'description' => $tag['description'] ?? '',
						'parent' => $tag['parent'] ?? 0,
					]);
				}
				if (!is_wp_error($term)) {
					$tag_slug_to_id[$tag['slug']] = (int)$term['term_id'];
				}
			}

			// --- 投稿ループ ---
			foreach ($data['all_posts'] ?? [] as $post_data) {
				// カテゴリID配列（slug → 新ID）
				$category_ids = [];
				foreach ($post_data['categories'] ?? [] as $cat) {
					if (isset($cat['slug'], $cat_slug_to_id[$cat['slug']])) {
						$category_ids[] = $cat_slug_to_id[$cat['slug']];
					}
				}

				// タグ名配列（slug → name）
				$tag_names = [];
				foreach ($post_data['tags'] ?? [] as $tag) {
					if (isset($tag['slug']) && isset($data['taxonomies']['post_tag'])) {
						foreach ($data['taxonomies']['post_tag'] as $term) {
							if ($term['slug'] === $tag['slug']) {
								$tag_names[] = $term['name'];
								break;
							}
						}
					}
				}

				// post_content 内の categories と tags のIDを置換
				$content = preg_replace_callback(
					'/"(categories|tags)":\s*\[(.*?)\]/s',
					function ($matches) use ($old_cat_id_to_slug, $cat_slug_to_id, $old_tag_id_to_slug, $tag_slug_to_id) {
						$taxonomy = $matches[1];
						$ids_raw = $matches[2];
						$ids = array_map(
							fn($s) => trim($s, "\" \t\n\r\0\x0B"),
							explode(',', $ids_raw)
						);

						$new_ids = [];

						if ($taxonomy === 'categories') {
							foreach ($ids as $old_id) {
								if (isset($old_cat_id_to_slug[$old_id])) {
									$slug = $old_cat_id_to_slug[$old_id];
									if (isset($cat_slug_to_id[$slug])) {
										$new_ids[] = '"' . $cat_slug_to_id[$slug] . '"';
										continue;
									}
								}
								$new_ids[] = '"' . $old_id . '"';
							}
						} elseif ($taxonomy === 'tags') {
							foreach ($ids as $old_id) {
								if (isset($old_tag_id_to_slug[$old_id])) {
									$slug = $old_tag_id_to_slug[$old_id];
									if (isset($tag_slug_to_id[$slug])) {
										$new_ids[] = '"' . $tag_slug_to_id[$slug] . '"';
										continue;
									}
								}
								$new_ids[] = '"' . $old_id . '"';
							}
						}

						return '"' . $taxonomy . '":[' . implode(',', $new_ids) . ']';
					},
					$post_data['post_content'] ?? ''
				);

				// 投稿登録
				wp_insert_post([
					'post_title'    => $post_data['post_title'] ?? '',
					'post_content'  => $content,
					'post_excerpt'  => $post_data['post_excerpt'] ?? '',
					'post_date'     => $post_data['post_date'] ?? '',
					'post_name'     => $post_data['post_name'] ?? '',
					'post_status'   => $post_data['post_status'] ?? 'publish',
					'post_author'   => $post_data['post_author'] ?? 1,
					'post_category' => $category_ids,
					'tags_input'    => $tag_names,
				]);
			}
		}

		public function import_pages($data) {
			$page_id_map = [];

			// 旧ID→slugマップ、slug→新IDマップは $data の taxonomies から作る想定
			$old_cat_id_to_slug = [];
			foreach ($data['taxonomies']['category'] ?? [] as $cat) {
				if (isset($cat['term_id'], $cat['slug'])) {
					$old_cat_id_to_slug[(string)$cat['term_id']] = $cat['slug'];
				}
			}
			$cat_slug_to_id = [];
			foreach ($data['taxonomies']['category'] ?? [] as $cat) {
				if (!isset($cat['slug'])) continue;
				$term = term_exists($cat['slug'], 'category');
				if (!$term) {
					$term = wp_insert_term($cat['name'], 'category', [
						'slug' => $cat['slug'],
						'description' => $cat['description'] ?? '',
						'parent' => $cat['parent'] ?? 0,
					]);
				}
				if (!is_wp_error($term)) {
					$cat_slug_to_id[$cat['slug']] = (int)$term['term_id'];
				}
			}

			$old_tag_id_to_slug = [];
			$tag_slug_to_id = [];
			foreach ($data['taxonomies']['post_tag'] ?? [] as $tag) {
				if (isset($tag['term_id'], $tag['slug'])) {
					$old_tag_id_to_slug[(string)$tag['term_id']] = $tag['slug'];
				}
				$term = term_exists($tag['slug'], 'post_tag');
				if (!$term) {
					$term = wp_insert_term($tag['name'], 'post_tag', [
						'slug' => $tag['slug'],
						'description' => $tag['description'] ?? '',
						'parent' => $tag['parent'] ?? 0,
					]);
				}
				if (!is_wp_error($term)) {
					$tag_slug_to_id[$tag['slug']] = (int)$term['term_id'];
				}
			}

			// ページループ
			foreach ($data['all_pages']['pages'] as $page_data) {
				// post_contentの categories, tags IDを置換
				$content = preg_replace_callback(
					'/"(categories|tags)":\s*\[(.*?)\]/s',
					function ($matches) use ($old_cat_id_to_slug, $cat_slug_to_id, $old_tag_id_to_slug, $tag_slug_to_id) {
						$taxonomy = $matches[1];
						$ids_raw = $matches[2];
						$ids = array_map(
							fn($s) => trim($s, "\" \t\n\r\0\x0B"),
							explode(',', $ids_raw)
						);

						$new_ids = [];

						if ($taxonomy === 'categories') {
							foreach ($ids as $old_id) {
								if (isset($old_cat_id_to_slug[$old_id])) {
									$slug = $old_cat_id_to_slug[$old_id];
									if (isset($cat_slug_to_id[$slug])) {
										$new_ids[] = '"' . $cat_slug_to_id[$slug] . '"';
										continue;
									}
								}
								$new_ids[] = '"' . $old_id . '"';
							}
						} elseif ($taxonomy === 'tags') {
							foreach ($ids as $old_id) {
								if (isset($old_tag_id_to_slug[$old_id])) {
									$slug = $old_tag_id_to_slug[$old_id];
									if (isset($tag_slug_to_id[$slug])) {
										$new_ids[] = '"' . $tag_slug_to_id[$slug] . '"';
										continue;
									}
								}
								$new_ids[] = '"' . $old_id . '"';
							}
						}

						return '"' . $taxonomy . '":[' . implode(',', $new_ids) . ']';
					},
					$page_data['post_content'] ?? ''
				);

				// 既存ページ確認
				$existing = get_page_by_path($page_data['post_name']);
				if ($existing) {
					wp_update_post([
						'ID'           => $existing->ID,
						'post_title'   => $page_data['post_title'],
						'post_content' => $content,
						'post_status'  => $page_data['post_status'],
						'menu_order'   => $page_data['menu_order'],
					]);
					$page_id_map[$page_data['post_name']] = $existing->ID;
				} else {
					$page_id = wp_insert_post([
						'post_type'    => 'page',
						'post_title'   => $page_data['post_title'],
						'post_content' => $content,
						'post_status'  => $page_data['post_status'],
						'post_name'    => $page_data['post_name'],
						'menu_order'   => $page_data['menu_order'],
					]);
					$page_id_map[$page_data['post_name']] = $page_id;
				}
			}

			// フロントページ設定
			if (isset($data['all_pages']['front_page']) && ($data['all_pages']['front_page']['show_on_front'] ?? '') === 'page') {
				$slug = $data['all_pages']['front_page']['page_on_front_slug'] ?? '';
				if ($slug && isset($page_id_map[$slug])) {
					update_option('show_on_front', 'page');
					update_option('page_on_front', $page_id_map[$slug]);
				}
			}
		}

		public function import_nishiki_pro_content( $data ){

			$id_map = [];

			foreach ($data['nishiki_pro_content_posts'] as $post_data) {

				//var_dump($post_data);

				$new_post_id = wp_insert_post([
					'post_title'   => $post_data['post_title'],
					'post_content' => $post_data['post_content'],
					'post_excerpt' => $post_data['post_excerpt'],
					'post_date'    => $post_data['post_date'],
					'post_name'    => $post_data['post_name'],
					'post_status'  => $post_data['post_status'],
					'post_author'  => $post_data['post_author'],
					'post_type'    => $post_data['post_type'],
				]);

				$id_map[$post_data['ID']] = $new_post_id;

			}

			// 全てのインポート後に nishiki_pro_content の select_content の ID を新しく置き換え
			foreach ( $data['content'] as &$section ) {
				foreach ( $section as &$area ) {
					if ( ! empty( $area['select_content'] ) ) {
						$old_id = intval( $area['select_content'] );
						if ( isset( $id_map[$old_id] ) ) {
							$area['select_content'] = (string)$id_map[$old_id];
						}
					}
				}
			}

			// 差し替えた nishiki_pro_content を新サイトにインポート
			foreach( $data['content'] as $key => $value ){
				update_option( 'nishiki_pro_content_' . $key, $value );
			}
		}

		public function import_nav_menus( $menus ){
			$locations = get_theme_mod('nav_menu_locations', []);

			foreach ($menus as $menu) {
				$menu_exists = wp_get_nav_menu_object($menu['slug']);
				if (!$menu_exists) {
					$menu_id = wp_create_nav_menu($menu['name']);

					$item_ids = [];

					foreach ($menu['items'] as $item) {
						$object_id = 0;

						// ID変換（投稿タイプ or タクソノミー）
						if ($item['type'] === 'post_type' && $item['slug'] && $item['object']) {
							$post = get_page_by_path($item['slug'], OBJECT, $item['object']);
							if ($post) {
								$object_id = $post->ID;
							}
						} elseif ($item['type'] === 'taxonomy' && $item['slug'] && $item['object']) {
							$term = get_term_by('slug', $item['slug'], $item['object']);
							if ($term) {
								$object_id = $term->term_id;
							}
						}

						// 基本情報
						$menu_args = [
							'menu-item-title'     => $item['title'],
							'menu-item-status'    => 'publish',
							'menu-item-parent-id' => 0,
							'menu-item-type'      => $item['type'],
							'menu-item-object'    => $item['object'],
							'menu-item-object-id' => $object_id,
						];

						// カスタムリンクはURL必須
						if ($item['type'] === 'custom') {
							$menu_args['menu-item-url'] = $item['url'];
						}

						$item_id = wp_update_nav_menu_item($menu_id, 0, $menu_args);
						$item_ids[$item['title']] = $item_id;
					}

					// 親子関係設定
					foreach ($menu['items'] as $item) {
						if (!empty($item['parent']) && isset($item_ids[$item['parent']])) {
							$child_id = $item_ids[$item['title']];
							$parent_id = $item_ids[$item['parent']];
							wp_update_nav_menu_item($menu_id, $child_id, [
								'menu-item-parent-id' => $parent_id,
							]);
						}
					}
				} else {
					$menu_id = $menu_exists->term_id;
				}

				// location設定（スキップせず必ず更新）
				if (!empty($menu['location'])) {
					$locations[$menu['location']] = $menu_id;
				}
			}

			// ロケーション設定を反映
			set_theme_mod('nav_menu_locations', $locations);
		}

		public function import_widgets( $widgets ){
			foreach ($widgets as $option_name => $option_value) {
				update_option($option_name, $option_value);
			}
		}

		public function import_theme_options( $data ){
			foreach( $data['theme_options'] as $key => $option_names ){
				if( $key !== 'content' && $key !== 'nishiki_pro_content_posts' ){
					foreach( $option_names as $option_name_key => $option_name_value ){               
						update_option( 'nishiki_pro_' . $key . '_' . $option_name_key, $option_name_value );
					}
				}
			}        
		}

		/**
		 * delete_all_data
		 */
		public function delete_all_data() {
			// 投稿と固定ページを削除
			$post_types = ['post', 'page', 'nishiki_pro_content'];
			foreach ($post_types as $type) {
				$posts = get_posts(['post_type' => $type, 'numberposts' => -1, 'post_status' => 'any']);
				foreach ($posts as $post) {
					wp_delete_post($post->ID, true);
				}
			}

			// sidebar
			delete_option('sidebars_widgets');

			// widget
			delete_option('widget_block');

			// navigation
			$this->delete_nav_menu();

			// theme_option
			$theme_options = [
				'general' => [
					'font',
					'optimize',
					'profile',
					'top',
				],
				'content' => [
					'common',
					'home',
					'post',
					'archive',
					'search',
					'404page',
				],
				'meta' => [
					'json_ld',
					'ogp',
				],
			];

			foreach( $theme_options as $key => $option_names ){
				foreach( $option_names as $option_name ){
					delete_option( 'nishiki_pro_' . $key . '_' . $option_name );
				}
			}

			// category & tag
			$this->delete_all_categories_and_tags();

			// theme_mods
			remove_theme_mods();
			
		}

		public function delete_nav_menu(){
			delete_option('nav_menu_options');

			$menus = get_terms( array(
				'taxonomy'   => 'nav_menu',
				'hide_empty' => false,
			) );

			if ( ! is_wp_error( $menus ) && ! empty( $menus ) ) {
				foreach ( $menus as $menu ) {
					if ( function_exists( 'wp_delete_nav_menu' ) ) {
						wp_delete_nav_menu( $menu->term_id );
					} else {
						wp_delete_term( $menu->term_id, 'nav_menu' );
					}
				}
			}
		}

		// カテゴリーとタグをすべて削除
		public function delete_all_categories_and_tags() {
			// 削除したいタクソノミー
			$taxonomies = array( 'category', 'post_tag' );

			foreach ( $taxonomies as $taxonomy ) {
				// 全タームを取得（空のタームも含む）
				$terms = get_terms( array(
					'taxonomy'   => $taxonomy,
					'hide_empty' => false,
				) );

				if ( ! is_wp_error( $terms ) && ! empty( $terms ) ) {
					foreach ( $terms as $term ) {
						// デフォルトカテゴリー（通常 ID=1）は削除しない方が安全
						if ( $taxonomy === 'category' && (int) $term->term_id === 1 ) {
							continue;
						}
						wp_delete_term( $term->term_id, $taxonomy );
					}
				}
			}
		}

		/**
		 * 既存コンテンツを全て初期化して選択したテンプレートを適用
		 */
		public function register_template( $topPage = '' ) {

			if ( ! current_user_can('administrator') ) return;

			
		}
	}

	new NISHIKI_PRO_SPDB_IMPORT_DEMO_REST_API();
}
